% Model of a steady state non-isothermal plug flow reactor - Gas phase

% Main file

% For more information, see:

% A. De Visscher 
% Lecture Notes in Chemical Engineering Kinetics and Chemical Reactor Design
% Createspace (2013) 

% (C) Alex De Visscher (2013)

% ---------------------------

% data

clear all

rhob = 450;          % Bed density of catalyst (kg/m3)
L = 4;               % Bed depth (m)
D = 0.05;            % Tube diameter (m)
Ntube = 300;         % Number of tubes
OMEGA = Ntube*(pi*D^2/4);  % Reactor cross sectional area
V = L*OMEGA;         % Reactor bed volume (m3)
W = V*rhob;          % Catalyst mass (kg)
Wspan = 0:W/100:W;   % We want output every 10 kg up to W kg
nE0 = 70;            % Initial alkene flow rate (mol/s)
nH0 = 10;            % Initial hydrogen flow rate (mol/s)
nA0 = 0;             % Initial alkane flow rate (mol/s)

           % Coefficients for rate constants k2, KE, KH, and KA
Ak2 = 4.8e6;         % Rate constant k2 (mol/bar1.5/kg/s)
Ek2 = 110000;
BKE = 6.5e-6;        % Rate constant KE (1/bar)
HKE = 55000;
BKH = 7.0e-4;        % Rate constant KH (1/bar)
HKH = 20000;
BKA = 8.5e-5;        % Rate constant KA (1/bar)
HKA = 38000;
p0 = 500000;          % Initial pressure (Pa)

ME = 0.084;          % Molar mass of E (kg/mol)
MH = 0.002;          % Molar mass of hydrogen
MA = 0.086;          % Molar mass of A
T0 = 663.15;         % Initial temperature (K)
Tc = 573.15;         % Cooling medium temperature (K)
dp = 0.006;          % Catalyst particle diameter (m)
mu = 2.5e-5;         % Gas viscosity (Pa s)
phi = 0.4;           % Bed porosity (-)

U = 60;              % Overall heat transfer coefficient (W/m2/K)
CpE = 246;           % Heat capacity of E (J/mol/K)
CpH = 29;            % Heat capacity of H2 (J/mol/K)
CpA = 266;           % Heat capacity of A (J/mol/K)
DH400 = -130000;     % Enthalply of reaction at 400 C

n0(1) = nE0;         % Running variable for nE is N(1)
n0(2) = nH0;         % Running variable for nH is N(2)
n0(3) = nA0;         % Running variable for nA is N(3)
n0(4) = p0;          % Running variable for p is N(4)
n0(5) = T0;          % Running variable for T is N(5)

% Call of differential equation solver

options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'InitialStep', 0.01);
[W,N] = ode45(@f,Wspan,n0,options,Ak2,Ek2,BKE,HKE,BKH,HKH,BKA,HKA,ME,MH,MA,D,OMEGA,mu,Tc,dp,rhob,phi,U,CpE,CpH,CpA,DH400);

% Output

figure(1)
plot(W,N(:,1:3))
xlabel('Catalyst mass (kg)');
ylabel('Molar flow rate (mol/s)');
figure(2)
plot(W,N(:,4))
xlabel('Catalyst mass (kg)');
ylabel('Pressure (Pa)');
figure(3)
plot(W,N(:,5))
xlabel('Catalyst mass (kg)');
ylabel('Temperature (K))');
W
N
[s1,s2] = size(W);
conversion = (N(1,2)-N(s1,2))/N(1,2)
